# JFrog Artifactory Helm Chart

## Prerequisites Details

* Kubernetes 1.6+
* Artifactory Pro trial license [get one from here](https://www.jfrog.com/artifactory/free-trial/)

## Chart Details
This chart will do the following:

* Deploy Artifactory-Pro/Artifactory-Edge (or OSS/CE if custom image is set)
* Deploy a PostgreSQL database using the stable/postgresql chart (can be changed)
* Deploy an optional Nginx server
* Optionally expose Artifactory with Ingress [Ingress documentation](https://kubernetes.io/docs/concepts/services-networking/ingress/)

## Installing the Chart

### Add JFrog Helm repository
Before installing JFrog helm charts, you need to add the [JFrog helm repository](https://charts.jfrog.io/) to your helm client
```bash
helm repo add jfrog https://charts.jfrog.io
```

### Install Chart
To install the chart with the release name `artifactory`:
```bash
helm install --name artifactory jfrog/artifactory
```

### Deploying Artifactory OSS
By default it will run Artifactory-Pro to run Artifactory-OSS use following command:
```bash
helm install --name artifactory --set artifactory.image.repository=docker.bintray.io/jfrog/artifactory-oss jfrog/artifactory
```

### Deploying Artifactory CE for C++
By default it will run Artifactory-Pro to run Artifactory-CE for C++ use following command:
```bash
helm install --name artifactory --set artifactory.image.repository=docker.bintray.io/jfrog/artifactory-cpp-ce jfrog/artifactory
```

### Deploying Artifactory with embedded Derby database
By default, this chart deploys Artifactory with PostgreSQL (running in a separate pod).
It's possible to deploy Artifactory without PostgreSQL (or any other external database), which will default to the embedded [Derby database](https://db.apache.org/derby/).
```bash
# Disable the default postgresql
helm install --name artifactory --set postgresql.enabled=false jfrog/artifactory
```
Artifactory will start with it's embedded Derby database.

### Deploying Artifactory with replicator
The [Artifactory replicator](https://www.jfrog.com/confluence/display/RTF/Replicator) is used with an [Enterprise Plus](https://www.jfrog.com/confluence/display/EP/Welcome+to+JFrog+Enterprise+Plus) license.
```bash
## Artifactory replicator is disabled by default. When the replicator is enabled, the replicator.publicUrl parameter is required. To enable it use the following:
helm install --name artifactory --set artifactory.replicator.enabled=true --set artifactory.replicator.publicUrl=<artifactory_url>:<replicator_port> jfrog/artifactory
```

### Deploying Artifactory for small/medium/large instllations
In the chart directory, we have added three values files, one for each installation type - small/medium/large. These values files are recommendations for setting resources requests and limits for your installation. The values are derived from the following [documentation](https://www.jfrog.com/confluence/display/EP/Installing+on+Kubernetes#InstallingonKubernetes-Systemrequirements). You can find them in the corresponding chart directory -  values-small.yaml, values-medium.yaml and values-large.yaml

### Accessing Artifactory
**NOTE:** It might take a few minutes for Artifactory's public IP to become available.
Follow the instructions outputted by the install command to get the Artifactory IP to access it.

### Updating Artifactory
Once you have a new chart version, you can update your deployment with
```bash
helm upgrade artifactory --namespace artifactory jfrog/artifactory
```

If artifactory was installed without providing a value to postgresql.postgresqlPassword (a password was autogenerated), follow these instructions:
1. Get the current password by running:
```bash
POSTGRES_PASSWORD=$(kubectl get secret -n <namespace> <myrelease>-postgresql -o jsonpath="{.data.postgresql-password}" | base64 --decode)
```
2. Upgrade the release by passing the previously auto-generated secret:
```bash
helm upgrade <myrelease> jfrog/artifactory --set postgresql.postgresqlPassword=${POSTGRES_PASSWORD}
```

This will apply any configuration changes on your existing deployment.

### Artifactory memory and CPU resources
The Artifactory Helm chart comes with support for configured resource requests and limits to Artifactory, Nginx and PostgreSQL. By default, these settings are commented out.
It is **highly** recommended to set these so you have full control of the allocated resources and limits.
Artifactory java memory parameters can (and should) also be set to match the allocated resources with `artifactory.javaOpts.xms` and `artifactory.javaOpts.xmx`.
```bash
# Example of setting resource requests and limits to all pods (including passing java memory settings to Artifactory)
helm install --name artifactory \
               --set artifactory.resources.requests.cpu="500m" \
               --set artifactory.resources.limits.cpu="2" \
               --set artifactory.resources.requests.memory="1Gi" \
               --set artifactory.resources.limits.memory="4Gi" \
               --set artifactory.javaOpts.xms="1g" \
               --set artifactory.javaOpts.xmx="4g" \
               --set nginx.resources.requests.cpu="100m" \
               --set nginx.resources.limits.cpu="250m" \
               --set nginx.resources.requests.memory="250Mi" \
               --set nginx.resources.limits.memory="500Mi" \
               jfrog/artifactory
```
Get more details on configuring Artifactory in the [official documentation](https://www.jfrog.com/confluence/).

### Artifactory storage
When using an enterprise license. Artifactory supports a wide range of storage back ends. You can see more details on [Artifactory Filestore options](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore)

In this chart, you set the type of storage you want with `artifactory.persistence.type` and pass the required configuration settings.
The default storage in this chart is the `file-system` replication, where the data is replicated to all nodes.

#### NFS
To use an NFS server as your cluster's storage, you need to
- Setup an NFS server. Get its IP as `NFS_IP`
- Create a `data` and `backup` directories on the NFS exported directory with write permissions to all
- Pass NFS parameters to `helm install` and `helm upgrade`
```bash
...
--set artifactory.persistence.type=nfs \
--set artifactory.persistence.nfs.ip=${NFS_IP} \
...
```

#### Google Storage
To use a Google Storage bucket as the cluster's filestore. See [Google Storage Binary Provider](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore#ConfiguringtheFilestore-GoogleStorageBinaryProvider)
- Pass Google Storage parameters to `helm install` and `helm upgrade`
```bash
...
--set artifactory.persistence.type=google-storage \
--set artifactory.persistence.googleStorage.identity=${GCP_ID} \
--set artifactory.persistence.googleStorage.credential=${GCP_KEY} \
...
```

#### AWS S3
**NOTE** Keep in mind that when using the `aws-s3` persistence type, you will not be able to provide an IAM on the pod level. 
In order to grant permissions to Artifactory using an IAM role, you will have to attach the said IAM role to the machine(s) on which Artifactory is running.
This is due to the fact that the `aws-s3` template uses the `JetS3t` library to interact with AWS. If you want to grant an IAM role at the pod level, see the `AWS S3 Vs` section.

To use an AWS S3 bucket as the cluster's filestore. See [S3 Binary Provider](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore#ConfiguringtheFilestore-S3BinaryProvider)
- Pass AWS S3 parameters to `helm install` and `helm upgrade`
```bash
...
# With explicit credentials:
--set artifactory.persistence.type=aws-s3 \
--set artifactory.persistence.awsS3.endpoint=${AWS_S3_ENDPOINT} \
--set artifactory.persistence.awsS3.region=${AWS_REGION} \
--set artifactory.persistence.awsS3.identity=${AWS_ACCESS_KEY_ID} \
--set artifactory.persistence.awsS3.credential=${AWS_SECRET_ACCESS_KEY} \
...

...
# With using existing IAM role
--set artifactory.persistence.type=aws-s3 \
--set artifactory.persistence.awsS3.endpoint=${AWS_S3_ENDPOINT} \
--set artifactory.persistence.awsS3.region=${AWS_REGION} \
--set artifactory.persistence.awsS3.roleName=${AWS_ROLE_NAME} \
...
```
**NOTE:** Make sure S3 `endpoint` and `region` match. See [AWS documentation on endpoint](https://docs.aws.amazon.com/general/latest/gr/rande.html)

#### AWS S3 V3
To use an AWS S3 bucket as the cluster's filestore and access it with the official AWS SDK, See [S3 Official SDK Binary Provider](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore#ConfiguringtheFilestore-AmazonS3OfficialSDKTemplate). 
This filestore template uses the official AWS SDK, unlike th`aws-s3` implementation that uses the `JetS3t` library.
Use this template if you want to attach an IAM role to the Artifactory pod directly (as opposed to attaching it to the machine/s that Artifactory will run on).

**NOTE** This will have to be combined with a k8s mechanism for attaching IAM roles to pods, like [kube2iam](https://github.com/helm/charts/tree/master/stable/kube2iam) or anything similar.
 
- Pass AWS S3 V3 parameters and the annotation pointing to the IAM role (when using an IAM role. this is kube2iam specific and may vary depending on the implementation) to `helm install` and `helm upgrade`

```bash
# With explicit credentials:
--set artifactory.persistence.type=aws-s3-v3 \
--set artifactory.persistence.awsS3V3.region=${AWS_REGION} \
--set artifactory.persistence.awsS3V3.bucketName=${AWS_S3_BUCKET_NAME} \
--set artifactory.persistence.awsS3V3.identity=${AWS_ACCESS_KEY_ID} \
--set artifactory.persistence.awsS3V3.credential=${AWS_SECRET_ACCESS_KEY} \
...
```

```bash
# With using existing IAM role
--set artifactory.persistence.type=aws-s3-v3 \
--set artifactory.persistence.awsS3V3.region=${AWS_REGION} \
--set artifactory.persistence.awsS3V3.bucketName=${AWS_S3_BUCKET_NAME} \
--set artifactory.annotations.'iam\.amazonaws\.com/role'=${AWS_IAM_ROLE_ARN}
...
```

#### Microsoft Azure Blob Storage
To use Azure Blob Storage as the cluster's filestore. See [Azure Blob Storage Binary Provider](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore#ConfiguringtheFilestore-AzureBlobStorageClusterBinaryProvider)
- Pass Azure Blob Storage parameters to `helm install` and `helm upgrade`
```bash
...
--set artifactory.persistence.type=azure-blob \
--set artifactory.persistence.azureBlob.accountName=${AZURE_ACCOUNT_NAME} \
--set artifactory.persistence.azureBlob.accountKey=${AZURE_ACCOUNT_KEY} \
--set artifactory.persistence.azureBlob.endpoint=${AZURE_ENDPOINT} \
--set artifactory.persistence.azureBlob.containerName=${AZURE_CONTAINER_NAME} \
...
```

* To use a persistent volume claim as cache dir together with Azure Blob Storage, additionally pass the following parameters to `helm  install` and `helm upgrade` (make sure `mountPath` and `cacheProviderDir` point to the same location)
```bash
...
--set artifactory.persistence.existingClaim=${YOUR_CLAIM} \
--set artifactory.persistence.mountPath=/opt/cache-dir \
--set artifactory.persistence.cacheProviderDir=/opt/cache-dir \
...
```

#### Custom binarystore.xml
You have an option to provide a custom [binarystore.xml](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore).<br>
There are two options for this

1. Editing directly in [values.yaml](values.yaml)
```yaml
artifactory:
  persistence:
    binarystoreXml: |
      <!-- The custom XML snippet -->
      <config version="v1">
          <chain template="file-system"/>
      </config>

```

2. Create your own [Secret](https://kubernetes.io/docs/concepts/configuration/secret/) and pass it to your `helm install` command
```yaml
# Prepare your custom Secret file (custom-binarystore.yaml)
kind: Secret
apiVersion: v1
metadata:
  name: custom-binarystore
  labels:
    app: artifactory
    chart: artifactory
stringData:
  binarystore.xml: |-
      <!-- The custom XML snippet -->
      <config version="v1">
          <chain template="file-system"/>
      </config>
```

```bash
# Create a secret from the file
kubectl apply -n artifactory -f ./custom-binarystore.yaml

# Pass it to your helm install command:
helm install --name artifactory --namespace artifactory --set artifactory.persistence.customBinarystoreXmlSecret=custom-binarystore jfrog/artifactory
```


### Customizing Database password
You can override the specified database password (set in [values.yaml](values.yaml)), by passing it as a parameter in the install command line
```bash
helm install --name artifactory --namespace artifactory --set postgresql.postgresqlPassword=12_hX34qwerQ2 jfrog/artifactory
```

You can customise other parameters in the same way, by passing them on `helm install` command line.

### Deleting Artifactory
```bash
helm delete --purge artifactory
```
This will completely delete your Artifactory Pro deployment.
**IMPORTANT:** This will also delete your data volumes. You will lose all data!

### Kubernetes Secret for Artifactory License
##### Use an existing secret
You can deploy the Artifactory license as a [Kubernetes secret](https://kubernetes.io/docs/concepts/configuration/secret/).
Prepare a text file with the license written in it and create a Kubernetes secret from it.
```bash
# Create the Kubernetes secret (assuming the local license file is 'art.lic')
kubectl create secret generic artifactory-license --from-file=./art.lic

# Pass the license to helm
helm install --name artifactory --set artifactory.license.secret=artifactory-license,artifactory.license.dataKey=art.lic jfrog/artifactory
```
**NOTE:** This method is relevant for initial deployment only! Once Artifactory is deployed, you should not keep passing these parameters as the license is already persisted into Artifactory's storage (they will be ignored).
Updating the license should be done via Artifactory UI or REST API.
If you want to keep managing the artifactory license using the same method, you can use the copyOnEveryStartup example shown in the values.yaml file
 
##### Create the secret as part of the helm release
values.yaml
```yaml
artifactory:
  license:
    licenseKey: |-
      <LICENSE_KEY>
```

```bash
helm install --name artifactory-ha -f values.yaml jfrog/artifactory-ha
```
**NOTE:** This method is relevant for initial deployment only! Once Artifactory is deployed, you should not keep passing these parameters as the license is already persisted into Artifactory's storage (they will be ignored).
Updating the license should be done via Artifactory UI or REST API.
If you want to keep managing the artifactory license using the same method, you can use the copyOnEveryStartup example shown in the values.yaml file


### Configure NetworkPolicy

NetworkPolicy specifies what ingress and egress is allowed in this namespace. It is encouraged to be more specific whenever possible to increase security of the system.

In the `networkpolicy` section of values.yaml you can specify a list of NetworkPolicy objects.

For podSelector, ingress and egress, if nothing is provided then a default `- {}` is applied which is to allow everything.

A full (but very wide open) example that results in 2 NetworkPolicy objects being created:
```yaml
networkpolicy:
  # Allows all ingress and egress to/from artifactory.
  - name: artifactory
    podSelector:
      matchLabels:
        app: artifactory
    egress:
    - {}
    ingress:
    - {}
  # Allows connectivity from artifactory pods to postgresql pods, but no traffic leaving postgresql pod.
  - name: postgres
    podSelector:
      matchLabels:
        app: postgresql
    ingress:
    - from:
      - podSelector:
          matchLabels:
            app: artifactory
```
### Artifactory JMX Configuration
** You can see some information about the exposed MBeans here - https://www.jfrog.com/confluence/display/RTF/Artifactory+JMX+MBeans

Enable JMX in your deployment:
```bash
helm install --name artifactory \
    --set artifactory.javaOpts.jmx.enabled=true \
    jfrog/artifactory
```
This will enable access to Artifactory with JMX on the default port (9010).
** You have the option to change the port by setting ```artifactory.javaOpts.jmx.port``` to your choice of port

In order to connect to Artifactory using JMX with jconsole (or any similar tool) installed on your computer, follow the following steps:
1. Enable JMX as described above and Change the Artifactory service to be of type LoadBalancer:
```bash
helm install --name artifactory \
    --set artifactory.javaOpts.jmx.enabled=true \
    --set artifactory.service.type=LoadBalancer \
    jfrog/artifactory 

``` 
2. The default setting for java.rmi.server.hostname is the service name (this is also configurable with ```artifactory.javaOpts.jmx.host```).
So in order to connect to Artifactory with jconsole you should map the Artifactory kuberentes service IP to the service name using your hosts file as such:
```
<artifactory-service-ip>    artifactory-<release-name>
```
3. Launch jconsole with the service address and port:
```bash
jconsole artifactory-<release-name>:<jmx-port>
```

### Access creds. bootstraping
**IMPORTANT:** Bootsrapping access creds. will automatically trigger a restart to the Artifactory pod and will allow access for the user `access-admin` from certain IP's.

* User guide to [bootstrap Artifactory Access credentials](https://www.jfrog.com/confluence/display/ACC/Configuring+Access)

1. Create `access-creds-values.yaml` and provide the IP (By default 127.0.0.1) and password:
```yaml
artifactory:
  accessAdmin:
    ip: "<IP_RANGE>" #Example: "*"
    password: "<PASSWD>"
```

2. Apply the `access-creds-values.yaml` file:
```bash
helm upgrade --install artifactory jfrog/artifactory -f access-creds-values.yaml
```
 
3. Restart Artifactory Pod (`Kubectl delete pod <pod_name>`)

### Bootstrapping Artifactory
**IMPORTANT:** Bootstrapping Artifactory needs license. Pass license as shown in above section.

* User guide to [bootstrap Artifactory Global Configuration](https://www.jfrog.com/confluence/display/RTF/Configuration+Files#ConfigurationFiles-BootstrappingtheGlobalConfiguration)
* User guide to [bootstrap Artifactory Security Configuration](https://www.jfrog.com/confluence/display/RTF/Configuration+Files#ConfigurationFiles-BootstrappingtheSecurityConfiguration)

1. Create `bootstrap-config.yaml` with artifactory.config.import.xml and security.import.xml as shown below:
```yaml
apiVersion: v1
kind: ConfigMap
metadata:
  name: my-release-bootstrap-config
data:
  artifactory.config.import.xml: |
    <config contents>
  security.import.xml: |
    <config contents>
```

2. Create configMap in Kubernetes:
```bash
kubectl apply -f bootstrap-config.yaml
```
3. Pass the configMap to helm
```bash
helm install --name artifactory --set artifactory.license.secret=artifactory-license,artifactory.license.dataKey=art.lic,artifactory.configMapName=my-release-bootstrap-config jfrog/artifactory
```
OR
```bash
helm install --name artifactory --set artifactory.license.licenseKey=<LICENSE_KEY>,artifactory.configMapName=my-release-bootstrap-config jfrog/artifactory
```

### Use custom nginx.conf with Nginx

Steps to create configMap with nginx.conf
* Create `nginx.conf` file.
```bash
kubectl create configmap nginx-config --from-file=nginx.conf
```
* Pass configMap to helm install
```bash
helm install --name artifactory --set nginx.customConfigMap=nginx-config jfrog/artifactory
```

### Use an external Database

#### PostgreSQL
There are cases where you will want to use an external PostgreSQL with a different database name e.g. `my-artifactory-db`, then you need set a custom PostgreSQL connection URL, where `my-artifactory-db` is the name of the database.

This can be done with the following parameters
```bash
...
--set postgresql.enabled=false \
--set database.type=postgresql \
--set database.url='jdbc:postgresql://${DB_HOST}:${DB_PORT}/my-artifactory-db' \
--set database.user=${DB_USER} \
--set database.password=${DB_PASSWORD} \
...
```
**NOTE:** You must set `postgresql.enabled=false` in order for the chart to use the `database.*` parameters. Without it, they will be ignored!

#### Other DB type
There are cases where you will want to use a different database and not the enclosed **PostgreSQL**.
See more details on [configuring the database](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Database)
> The official Artifactory Docker images include the PostgreSQL database driver.
> For other database types, you will have to add the relevant database driver to Artifactory's tomcat/lib

This can be done with the following parameters
```bash
# Make sure your Artifactory Docker image has the MySQL database driver in it
...
--set postgresql.enabled=false \
--set artifactory.preStartCommand="wget -O /opt/jfrog/artifactory/tomcat/lib/mysql-connector-java-5.1.41.jar https://jcenter.bintray.com/mysql/mysql-connector-java/5.1.41/mysql-connector-java-5.1.41.jar" \
--set database.type=mysql \
--set database.host=${DB_HOST} \
--set database.port=${DB_PORT} \
--set database.user=${DB_USER} \
--set database.password=${DB_PASSWORD} \
...
```
**NOTE:** You must set `postgresql.enabled=false` in order for the chart to use the `database.*` parameters. Without it, they will be ignored!

#### Using pre-existing Kubernetes Secret
If you store your database credentials in a pre-existing Kubernetes `Secret`, you can specify them via `database.secrets` instead of `database.user` and `database.password`:
```bash
# Create a secret containing the database credentials
kubectl create secret generic my-secret --from-literal=user=${DB_USER} --from-literal=password=${DB_PASSWORD}
...
--set postgresql.enabled=false \
--set database.secrets.user.name=my-secret \
--set database.secrets.user.key=user \
--set database.secrets.password.name=my-secret \
--set database.secrets.password.key=password \
--set database.secrets.url.name=my-secret \
--set database.secrets.url.key=url \
...
```

### Deleting Artifactory
To delete the Artifactory.
```bash
helm delete --purge artifactory
```
This will completely delete your Artifactory HA cluster.

### Custom Docker registry for your images
If you need to pull your Docker images from a private registry, you need to create a
[Kubernetes Docker registry secret](https://kubernetes.io/docs/tasks/configure-pod-container/pull-image-private-registry/) and pass it to helm
```bash
# Create a Docker registry secret called 'regsecret'
kubectl create secret docker-registry regsecret --docker-server=<your-registry-server> --docker-username=<your-name> --docker-password=<your-pword> --docker-email=<your-email>
```
Once created, you pass it to `helm`
```bash
helm install --name artifactory --set imagePullSecrets=regsecret jfrog/artifactory
```

### Logger sidecars
This chart provides the option to add sidecars to tail various logs from Artifactory. See the available values in [values.yaml](values.yaml)

Get list of containers in the pod
```bash
kubectl get pods -n <NAMESPACE> <POD_NAME> -o jsonpath='{.spec.containers[*].name}' | tr ' ' '\n'
```

View specific log
```bash
kubectl logs -n <NAMESPACE> <POD_NAME> -c <LOG_CONTAINER_NAME>
```

### Custom init containers
There are cases where a special, unsupported init processes is needed like checking something on the file system or testing something before spinning up the main container.

For this, there is a section for writing a custom init container in the [values.yaml](values.yaml). By default it's commented out
```yaml
artifactory:
  ## Add custom init containers
  customInitContainers: |
    ## Init containers template goes here ##
```

### Custom sidecar containers
There are cases where an extra sidecar container is needed. For example monitoring agents or log collection.

For this, there is a section for writing a custom sidecar container in the [values.yaml](values.yaml). By default it's commented out
```yaml
artifactory:
  ## Add custom sidecar containers
  customSidecarContainers: |
    ## Sidecar containers template goes here ##
```

### Custom volumes
If you need to use a custom volume in a custom init or sidecar container, you can use this option.

For this, there is a section for defining custom volumes in the [values.yaml](values.yaml). By default it's commented out
```yaml
artifactory:
  ## Add custom volumes
  customVolumes: |
    ## Custom volume comes here ##
```

You can configure the sidecar to run as a custom user if needed by setting the following in the container template
```yaml
  # Example of running container as root (id 0)
  securityContext:
    runAsUser: 0
    fsGroup: 0
```

### Add Artifactory User Plugin during installation
If you need to add [Artifactory User Plugin](https://github.com/jfrog/artifactory-user-plugins), you can use this option.

Create a secret with [Artifactory User Plugin](https://github.com/jfrog/artifactory-user-plugins) by following command:
```bash
# Secret with single user plugin
kubectl  create secret generic archive-old-artifacts --from-file=archiveOldArtifacts.groovy --namespace=artifactory 

# Secret with single user plugin with configuration file
kubectl  create secret generic webhook --from-file=webhook.groovy  --from-file=webhook.config.json.sample --namespace=artifactory
```

Add plugin secret names to `plugins.yaml` as following:
```yaml
artifactory:
  userPluginSecrets:
    - archive-old-artifacts
    - webhook
```

You can now pass the created `plugins.yaml` file to helm install command to deploy Artifactory with user plugins as follows:
```bash
helm install --name artifactory -f plugins.yaml jfrog/artifactory
```

Alternatively, you may be in a situation in which you would like to create a secret in a Helm chart that depends on this chart. In this scenario, the name of the secret is likely dynamically generated via template functions, so passing a statically named secret isn't possible. In this case, the chart supports evaluating strings as templates via the [`tpl`](https://helm.sh/docs/charts_tips_and_tricks/#using-the-tpl-function) function - simply pass the raw string containing the templating language used to name your secret as a value instead by adding the following to your chart's `values.yaml` file:
```yaml
artifactory: # Name of the artifactory dependency
  artifactory:
    userPluginSecrets:
      - '{{ template "my-chart.fullname" . }}'
```
NOTE: By defining userPluginSecrets, this overrides any pre-defined plugins from the container image that are stored in /tmp/plugins.  At this time [artifactory-pro:6.9.0](https://bintray.com/jfrog/artifactory-pro) is distributed with `internalUser.groovy` plugin.  If you need this plugin in addition to your user plugins, you should include these additional plugins as part of your userPluginSecrets.

## Configuration
The following table lists the configurable parameters of the artifactory chart and their default values.

|         Parameter         |           Description             |                         Default                          |
|---------------------------|-----------------------------------|----------------------------------------------------------|
| `imagePullSecrets`        | Docker registry pull secret       |                                                          |
| `serviceAccount.create`   | Specifies whether a ServiceAccount should be created | `true`                                |
| `serviceAccount.name`     | The name of the ServiceAccount to create             | Generated using the fullname template |
| `serviceAccount.annotations`     | Artifactory service account annotations       | `` |
| `rbac.create`             | Specifies whether RBAC resources should be created   | `true`                                |
| `rbac.role.rules`         | Rules to create                   | `[]`                                                     |
| `logger.image.repository` | repository for logger image       | `busybox`                                                |
| `logger.image.tag`        | tag for logger image              | `1.30`                                                   |
| `artifactory.name`        | Artifactory name                  | `artifactory`                                            |
| `artifactory.replicaCount`            | Replica count for Artifactory deployment| `1`                                    |
| `artifactory.image.pullPolicy`         | Container pull policy             | `IfNotPresent`                              |
| `artifactory.image.repository`    | Container image                   | `docker.bintray.io/jfrog/artifactory-pro`        |
| `artifactory.image.version`       | Container tag                     |  `.Chart.AppVersion`                             |
| `artifactory.labels`              | Artifactory labels                | `{}`                                             |
| `artifactory.loggers`             | Artifactory loggers (see values.yaml for possible values) | `[]`                     |
| `artifactory.catalinaLoggers`     | Artifactory Tomcat loggers (see values.yaml for possible values) | `[]`              |
| `artifactory.customInitContainers`| Custom init containers            |                                                  |
| `artifactory.customSidecarContainers`| Custom sidecar containers      |                                                  |
| `artifactory.customVolumes`       | Custom volumes                    |                                                  |
| `artifactory.customVolumeMounts`  | Custom Artifactory volumeMounts   |                                                  |
| `artifactory.userPluginSecrets`   | Array of secret names for Artifactory user plugins |                                 |
| `artifactory.license.licenseKey` | Artifactory license key. Providing the license key as a parameter will cause a secret containing the license key to be created as part of the release. Use either this setting or the license.secret and license.dataKey. If you use both, the latter will be used.  |           |
| `artifactory.license.secret` | Artifactory license secret name              |                                            |
| `artifactory.license.dataKey`| Artifactory license secret data key          |                                            |
| `artifactory.service.name`| Artifactory service name to be set in Nginx configuration | `artifactory`                    |
| `artifactory.service.type`| Artifactory service type | `ClusterIP`                                                       |
| `artifactory.service.loadBalancerSourceRanges`| Artifactory service array of IP CIDR ranges to whitelist (only when service type is LoadBalancer) |  |
| `artifactory.service.annotations` | Artifactory service annotations           | `{}`                            |
| `artifactory.externalPort` | Artifactory service external port | `8081`                                                  |
| `artifactory.internalPort` | Artifactory service internal port | `8081`                                                  |
| `artifactory.internalPortReplicator` | Replicator service internal port | `6061`                                         |
| `artifactory.externalPortReplicator` | Replicator service external port | `6061`                                         |
| `artifactory.livenessProbe.enabled`              | Enable liveness probe                     | `true`                    |
| `artifactory.livenessProbe.path`                     | Liveness probe HTTP Get path                           |  `/artifactory/webapp/#/login` |
| `artifactory.livenessProbe.initialDelaySeconds`  | Delay before liveness probe is initiated  | 180                       |
| `artifactory.livenessProbe.periodSeconds`        | How often to perform the probe            | 10                        |
| `artifactory.livenessProbe.timeoutSeconds`       | When the probe times out                  | 10                        |
| `artifactory.livenessProbe.successThreshold`     | Minimum consecutive successes for the probe to be considered successful after having failed. | 1 |
| `artifactory.livenessProbe.failureThreshold`     | Minimum consecutive failures for the probe to be considered failed after having succeeded.   | 10 |
| `artifactory.masterKey`                          | Artifactory masterkey. A 128-Bit key size (hexadecimal encoded) string (32 hex characters). Can be generated with `openssl rand -hex 16`.
NOTE: This key is generated only once and cannot be updated once created | `` |
| `artifactory.masterKeySecretName`                | Artifactory Master Key secret name |                                                                    |
| `artifactory.accessAdmin.ip`                     | Artifactory access-admin ip to be set upon startup, can use (*) for 0.0.0.0| 127.0.0.1                                    |
| `artifactory.accessAdmin.password`               | Artifactory access-admin password to be set upon startup|                                               |
| `artifactory.accessAdmin.secret`                 | Artifactory access-admin secret name |                                                                    |
| `artifactory.accessAdmin.dataKey`                | Artifactory access-admin secret data key |                                                                    |
| `artifactory.preStartCommand`                    | Command to run before entrypoint starts |                             |
| `artifactory.postStartCommand`                   | Command to run after container starts   |                             |
| `artifactory.extraEnvironmentVariables`          | Extra environment variables to pass to Artifactory. Supports evaluating strings as templates via the [`tpl`](https://helm.sh/docs/charts_tips_and_tricks/#using-the-tpl-function) function. See [documentation](https://www.jfrog.com/confluence/display/RTF/Installing+with+Docker#InstallingwithDocker-SupportedEnvironmentVariables) |   |
| `artifactory.readinessProbe.enabled`             | would you like a readinessProbe to be enabled           |  `true`     |
| `artifactory.readinessProbe.path`                | Readiness probe HTTP Get path                           |  `/artifactory/webapp/#/login` |
| `artifactory.readinessProbe.initialDelaySeconds` | Delay before readiness probe is initiated | 60                        |
| `artifactory.readinessProbe.periodSeconds`       | How often to perform the probe            | 10                        |
| `artifactory.readinessProbe.timeoutSeconds`      | When the probe times out                  | 10                        |
| `artifactory.readinessProbe.successThreshold`    | Minimum consecutive successes for the probe to be considered successful after having failed. | 1 |
| `artifactory.readinessProbe.failureThreshold`    | Minimum consecutive failures for the probe to be considered failed after having succeeded.   | 10 |
| `artifactory.deleteDBPropertiesOnStartup`    | Whether to delete the ARTIFACTORY_HOME/etc/db.properties file on startup. Disabling this will remove the ability for the db.properties to be updated with any DB-related environment variables change (e.g. DB_HOST, DB_URL)  | `true` |
| `artifactory.copyOnEveryStartup`         | List of files to copy on startup from source (which is absolute) to target (which is relative to ARTIFACTORY_HOME   |  |
| `artifactory.persistence.mountPath`      | Artifactory persistence volume mount path        | `"/var/opt/jfrog/artifactory"`       |
| `artifactory.persistence.enabled`        | Artifactory persistence volume enabled           | `true`                               |
| `artifactory.persistence.existingClaim`  | Artifactory persistence volume claim name        |                                      |
| `artifactory.persistence.accessMode`     | Artifactory persistence volume access mode       | `ReadWriteOnce`                      |
| `artifactory.persistence.size`           | Artifactory persistence or local volume size     | `20Gi`                               |
| `artifactory.persistence.binarystoreXml` | Artifactory binarystore.xml template             | See `values.yaml`                    |
| `artifactory.persistence.customBinarystoreXmlSecret` | A custom Secret for binarystore.xml  | ``                                   |
| `artifactory.persistence.maxCacheSize`   | The maximum storage allocated for the cache in bytes. | `50000000000`                   |
| `artifactory.persistence.cacheProviderDir`  | the root folder of binaries for the filestore cache. If the value specified starts with a forward slash ("/") it is considered the fully qualified path to the filestore folder. Otherwise, it is considered relative to the *baseDataDir*. | `cache`                   |
| `artifactory.persistence.type`              | Artifactory HA storage type                         | `file-system`                   |
| `artifactory.persistence.redundancy`        | Artifactory HA storage redundancy                   | `3`                             |
| `artifactory.persistence.nfs.ip`            | NFS server IP                        |                                     |
| `artifactory.persistence.nfs.haDataMount`   | NFS data directory                   | `/data`                             |
| `artifactory.persistence.nfs.haBackupMount` | NFS backup directory                 | `/backup`                           |
| `artifactory.persistence.nfs.dataDir`       | HA data directory                    | `/var/opt/jfrog/artifactory`     |
| `artifactory.persistence.nfs.backupDir`     | HA backup directory                  | `/var/opt/jfrog/artifactory-backup` |
| `artifactory.persistence.nfs.capacity`      | NFS PVC size                         | `200Gi`                             |
| `artifactory.persistence.eventual.numberOfThreads`  | Eventual number of threads   | `10`                                |
| `artifactory.persistence.googleStorage.endpoint`    | Google Storage API endpoint| `storage.googleapis.com`             |
| `artifactory.persistence.googleStorage.httpsOnly`   | Google Storage API has to be consumed https only| `false`             |
| `artifactory.persistence.googleStorage.bucketName`  | Google Storage bucket name          | `artifactory`             |
| `artifactory.persistence.googleStorage.identity`    | Google Storage service account id   |                              |
| `artifactory.persistence.googleStorage.credential`  | Google Storage service account key  |                              |
| `artifactory.persistence.googleStorage.path`        | Google Storage path in bucket       | `artifactory/filestore`   |
| `artifactory.persistence.googleStorage.bucketExists`| Google Storage bucket exists therefore does not need to be created.| `false`             |
| `artifactory.persistence.awsS3.bucketName`          | AWS S3 bucket name                     | `artifactory-aws`             |
| `artifactory.persistence.awsS3.endpoint`            | AWS S3 bucket endpoint                 | See https://docs.aws.amazon.com/general/latest/gr/rande.html |
| `artifactory.persistence.awsS3.region`              | AWS S3 bucket region                   |                              |
| `artifactory.persistence.awsS3.roleName`            | AWS S3 IAM role name                   |                             |
| `artifactory.persistence.awsS3.identity`            | AWS S3 AWS_ACCESS_KEY_ID               |                              |
| `artifactory.persistence.awsS3.credential`          | AWS S3 AWS_SECRET_ACCESS_KEY           |                              |
| `artifactory.persistence.awsS3.properties`          | AWS S3 additional properties           |                              |
| `artifactory.persistence.awsS3.path`                | AWS S3 path in bucket                  | `artifactory/filestore`   |
| `artifactory.persistence.awsS3.refreshCredentials`  | AWS S3 renew credentials on expiration | `true` (When roleName is used, this parameter will be set to true) |
| `artifactory.persistence.awsS3.httpsOnly`  | AWS S3 https access to the bucket only | `true` |
| `artifactory.persistence.awsS3.testConnection`      | AWS S3 test connection on start up     | `false`                      |
| `artifactory.persistence.awsS3.s3AwsVersion`        | AWS S3 signature version               | `AWS4-HMAC-SHA256`                      |
| `artifactory.persistence.awsS3V3.testConnection`        | AWS S3 test connection on start up               | `false`                      |
| `artifactory.persistence.awsS3V3.identity`            | AWS S3 AWS_ACCESS_KEY_ID               |                              |
| `artifactory.persistence.awsS3V3.credential`          | AWS S3 AWS_SECRET_ACCESS_KEY           |                              |
| `artifactory.persistence.awsS3V3.region`        | AWS S3 bucket region               |                       |
| `artifactory.persistence.awsS3V3.bucketName`        | AWS S3 bucket name               | `artifactory-aws`                      |
| `artifactory.persistence.awsS3V3.path`                | AWS S3 path in bucket                  | `artifactory/filestore`   |
| `artifactory.persistence.awsS3V3.endpoint`            | AWS S3 bucket endpoint                 | See https://docs.aws.amazon.com/general/latest/gr/rande.html |
| `artifactory.persistence.awsS3V3.kmsServerSideEncryptionKeyId`        | AWS S3 encryption key ID or alias  |              |
| `artifactory.persistence.awsS3V3.kmsKeyRegion`        | AWS S3 KMS Key region  |              |
| `artifactory.persistence.awsS3V3.kmsCryptoMode`        | AWS S3 KMS encryption mode  | See https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore#ConfiguringtheFilestore-AmazonS3OfficialSDKTemplate |
| `artifactory.persistence.awsS3V3.useInstanceCredentials`        | AWS S3 Use default authentication mechanism  | See https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore#ConfiguringtheFilestore-authentication             |
| `artifactory.persistence.awsS3V3.usePresigning`        | AWS S3 Use URL signing  | `false`            |
| `artifactory.persistence.awsS3V3.signatureExpirySeconds`        | AWS S3 Validity period in seconds for signed URLs  | `300`            |
| `artifactory.persistence.awsS3V3.cloudFrontDomainName`        | AWS CloudFront domain name  | See https://www.jfrog.com/confluence/display/RTF/Direct+Cloud+Storage+Download#DirectCloudStorageDownload-UsingCloudFront(Optional)|
| `artifactory.persistence.awsS3V3.cloudFrontKeyPairId`        | AWS CloudFront key pair ID  | See https://www.jfrog.com/confluence/display/RTF/Direct+Cloud+Storage+Download#DirectCloudStorageDownload-UsingCloudFront(Optional)|
| `artifactory.persistence.awsS3V3.cloudFrontPrivateKey`        | AWS CloudFront private key  | See https://www.jfrog.com/confluence/display/RTF/Direct+Cloud+Storage+Download#DirectCloudStorageDownload-UsingCloudFront(Optional)|
| `artifactory.persistence.azureBlob.accountName`     | Azure Blob Storage account name        | ``                        |
| `artifactory.persistence.azureBlob.accountKey`      | Azure Blob Storage account key         | ``                        |
| `artifactory.persistence.azureBlob.endpoint`        | Azure Blob Storage endpoint            | ``                        |
| `artifactory.persistence.azureBlob.containerName`   | Azure Blob Storage container name      | ``                        |
| `artifactory.persistence.azureBlob.testConnection`  | Azure Blob Storage test connection     | `false`                   |
| `artifactory.resources.requests.memory` | Artifactory initial memory request                  |                          |
| `artifactory.resources.requests.cpu`    | Artifactory initial cpu request     |                                          |
| `artifactory.resources.limits.memory`   | Artifactory memory limit            |                                          |
| `artifactory.resources.limits.cpu`      | Artifactory cpu limit               |                                          |
| `artifactory.javaOpts.xms`              | Artifactory java Xms size           |                                          |
| `artifactory.javaOpts.xmx`              | Artifactory java Xms size           |                                          |
| `artifactory.javaOpts.corePoolSize` | The number of async processes that can run in parallel - https://jfrog.com/knowledge-base/how-do-i-tune-artifactory-for-heavy-loads/  |   `8` |
| `artifactory.javaOpts.jmx.enabled`              | Enable JMX monitoring           |  `false`                                        |
| `artifactory.javaOpts.jmx.port`              | JMX Port number            |  `9010`                                        |
| `artifactory.javaOpts.jmx.host`              | JMX hostname (parsed as a helm template)           |  `{{ template "artifactory.fullname" $ }}` |
| `artifactory.javaOpts.jmx.ssl`              | Enable SSL           |  `false` |
| `artifactory.javaOpts.other`            | Artifactory additional java options |                                          |
| `artifactory.replicator.enabled`            | Enable Artifactory Replicator | `false`                                    |
| `artifactory.replicator.publicUrl`            | Artifactory Replicator Public URL |                                      |
| `artifactory.terminationGracePeriodSeconds`   | Termination grace period (seconds)  | `30s`                                |
| `ingress.enabled`           | If true, Artifactory Ingress will be created | `false`                                     |
| `ingress.annotations`       | Artifactory Ingress annotations     | `{}`                                                 |
| `ingress.labels`       | Artifactory Ingress labels     | `{}`                                                           |
| `ingress.hosts`             | Artifactory Ingress hostnames       | `[]`                                                 |
| `ingress.path`              | Artifactory Ingress path            | `/`                                                  |
| `ingress.tls`               | Artifactory Ingress TLS configuration (YAML) | `[]`                                        |
| `ingress.defaultBackend.enabled` | If true, the default `backend` will be added using serviceName and servicePort | `true` |
| `ingress.annotations`       | Ingress annotations, which are written out if annotations section exists in values. Everything inside of the annotations section will appear verbatim inside the resulting manifest. See `Ingress annotations` section below for examples of how to leverage the annotations, specifically for how to enable docker authentication. |  |
| `ingress.additionalRules`       | Ingress additional rules to be added to the Artifactory ingress. | `[]`  |
| `nginx.name` | Nginx name | `nginx`                                                                                      |
| `nginx.enabled` | Deploy nginx server | `true`                                                                           |
| `nginx.replicaCount` | Nginx replica count | `1`                                                                         |
| `nginx.uid`                 | Nginx User Id                     | `104`                                                  |
| `nginx.gid`                 | Nginx Group Id                    | `107`                                                  |
| `nginx.image.repository`    | Container image                   | `docker.bintray.io/jfrog/nginx-artifactory-pro`        |
| `nginx.image.version`       | Container tag                     | `.Chart.AppVersion`                                    |
| `nginx.image.pullPolicy`    | Container pull policy                   | `IfNotPresent`                                   |
| `nginx.labels`              | Nginx deployment labels           | `{}`                                                   |
| `nginx.loggers`        | Artifactory loggers (see values.yaml for possible values) | `[]`                           |
| `nginx.mainConf`        | Content of the Artifactory nginx main nginx.conf config file | `see values.yaml`                           |
| `nginx.artifactoryConf`        | Content of Artifactory nginx artifactory.conf config file | `see values.yaml`                           |
| `nginx.service.type`| Nginx service type | `LoadBalancer`                                                                |
| `nginx.service.loadBalancerSourceRanges`| Nginx service array of IP CIDR ranges to whitelist (only when service type is LoadBalancer) |  |
| `nginx.service.externalTrafficPolicy`| Nginx service desires to route external traffic to node-local or cluster-wide endpoints. | `Cluster` |
| `nginx.loadBalancerIP`| Provide Static IP to configure with Nginx |                                                      |
| `nginx.externalPortHttp` | Nginx service external port | `80`                                                            |
| `nginx.internalPortHttp` | Nginx service internal port | `80`                                                            |
| `nginx.externalPortHttps` | Nginx service external port | `443`                                                          |
| `nginx.internalPortHttps` | Nginx service internal port | `443`                                                          |
| `nginx.internalPortReplicator` | Replicator service internal port | `6061`                                               |
| `nginx.externalPortReplicator` | Replicator service external port | `6061`                                               |
| `nginx.livenessProbe.enabled`              | Enable liveness probe                     | `true`                          |
| `nginx.livenessProbe.path`                     | Liveness probe HTTP Get path                           |  `/artifactory/webapp/#/login` |
| `nginx.livenessProbe.initialDelaySeconds`  | Delay before liveness probe is initiated  | 60                              |
| `nginx.livenessProbe.periodSeconds`        | How often to perform the probe            | 10                              |
| `nginx.livenessProbe.timeoutSeconds`       | When the probe times out                  | 10                              |
| `nginx.livenessProbe.successThreshold`     | Minimum consecutive successes for the probe to be considered successful after having failed. | 10 |
| `nginx.livenessProbe.failureThreshold`     | Minimum consecutive failures for the probe to be considered failed after having succeeded.   | 1|
| `nginx.readinessProbe.enabled`              | would you like a readinessProbe to be enabled           |  `true`          |
| `nginx.readinessProbe.path`                     | Readiness probe HTTP Get path                           |  `/artifactory/webapp/#/login` |
| `nginx.readinessProbe.initialDelaySeconds` | Delay before readiness probe is initiated | 60                              |
| `nginx.readinessProbe.periodSeconds`       | How often to perform the probe            | 10                              |
| `nginx.readinessProbe.timeoutSeconds`      | When the probe times out                  | 10                              |
| `nginx.readinessProbe.successThreshold`    | Minimum consecutive successes for the probe to be considered successful after having failed. | 10 |
| `nginx.readinessProbe.failureThreshold`    | Minimum consecutive failures for the probe to be considered failed after having succeeded.   | 1 |
| `nginx.tlsSecretName` |  SSL secret that will be used by the Nginx pod |                                                 |
| `nginx.customConfigMap`           | Nginx CustomeConfigMap name for `nginx.conf` | ` `                                   |
| `nginx.customArtifactoryConfigMap`| Nginx CustomeConfigMap name for `artifactory.conf` | ` `                             |
| `nginx.persistence.mountPath` | Nginx persistence volume mount path | `"/var/opt/jfrog/nginx"`                           |
| `nginx.persistence.enabled` | Nginx persistence volume enabled | `false`                                                  |
| `nginx.persistence.accessMode` | Nginx persistence volume access mode | `ReadWriteOnce`                                  |
| `nginx.persistence.size` | Nginx persistence volume size | `5Gi`                                                         |
| `nginx.resources.requests.memory` | Nginx initial memory request  |                                                      |
| `nginx.resources.requests.cpu`    | Nginx initial cpu request     |                                                      |
| `nginx.resources.limits.memory`   | Nginx memory limit            |                                                      |
| `nginx.resources.limits.cpu`      | Nginx cpu limit               |                                                      |
| `waitForDatabase`                 | Wait for database (using wait-for-db init container)  | `true`                       |
| `postgresql.enabled`              | Use enclosed PostgreSQL as database        | `true`                                  |
| `postgresql.imageTag`             | PostgreSQL version                         | `9.6.11`                                |
| `postgresql.postgresqlDatabase`   | PostgreSQL database name                   | `artifactory`                           |
| `postgresql.postgresqlUsername`   | PostgreSQL database user                   | `artifactory`                           |
| `postgresql.postgresqlPassword`   | PostgreSQL database password               |                                         |
| `postgresql.persistence.enabled`  | PostgreSQL use persistent storage          | `true`                                  |
| `postgresql.persistence.size`     | PostgreSQL persistent storage size         | `50Gi`                                  |
| `postgresql.service.port`         | PostgreSQL database port                   | `5432`                                  |
| `postgresql.resources.requests.memory`    | PostgreSQL initial memory request  |                                         |
| `postgresql.resources.requests.cpu`       | PostgreSQL initial cpu request     |                                         |
| `postgresql.resources.limits.memory`      | PostgreSQL memory limit            |                                         |
| `postgresql.resources.limits.cpu`         | PostgreSQL cpu limit               |                                         |
| `database.type`                  | External database type (`postgresql`, `mysql`, `oracle` or `mssql`)  |                       |
| `database.host`                  | External database hostname                         |                                         |
| `database.port`                  | External database port                             |                                         |
| `database.url`                   | External database connection URL                   |                                         |
| `database.user`                  | External database username                         |                                         |
| `database.password`              | External database password                         |                                         |
| `database.secrets.user.name`     | External database username `Secret` name           |                                         |
| `database.secrets.user.key`      | External database username `Secret` key            |                                         |
| `database.secrets.password.name` | External database password `Secret` name           |                                         |
| `database.secrets.password.key`  | External database password `Secret` key            |                                         |
| `database.secrets.url.name     ` | External database url `Secret` name                |                                         |
| `database.secrets.url.key`       | External database url `Secret` key                 |                                         |
| `networkpolicy.name`             | Becomes part of the NetworkPolicy object name                                  | `artifactory`                           |
| `networkpolicy.podselector`      | Contains the YAML that specifies how to match pods. Usually using matchLabels. |                                         |
| `networkpolicy.ingress`          | YAML snippet containing to & from rules applied to incoming traffic            | `- {}` (open to all inbound traffic)    |
| `networkpolicy.egress`           | YAML snippet containing to & from rules applied to outgoing traffic            | `- {}` (open to all outbound traffic)   |

Specify each parameter using the `--set key=value[,key=value]` argument to `helm install`.

### Ingress and TLS
To get Helm to create an ingress object with a hostname, add these two lines to your Helm command:
```bash
helm install --name artifactory \
  --set ingress.enabled=true \
  --set ingress.hosts[0]="artifactory.company.com" \
  --set artifactory.service.type=NodePort \
  --set nginx.enabled=false \
  jfrog/artifactory
```

If your cluster allows automatic creation/retrieval of TLS certificates (e.g. [cert-manager](https://github.com/jetstack/cert-manager)), please refer to the documentation for that mechanism.

To manually configure TLS, first create/retrieve a key & certificate pair for the address(es) you wish to protect. Then create a TLS secret in the namespace:

```bash
kubectl create secret tls artifactory-tls --cert=path/to/tls.cert --key=path/to/tls.key
```

Include the secret's name, along with the desired hostnames, in the Artifactory Ingress TLS section of your custom `values.yaml` file:

```yaml
  ingress:
    ## If true, Artifactory Ingress will be created
    ##
    enabled: true

    ## Artifactory Ingress hostnames
    ## Must be provided if Ingress is enabled
    ##
    hosts:
      - artifactory.domain.com
    annotations:
      kubernetes.io/tls-acme: "true"
    ## Artifactory Ingress TLS configuration
    ## Secrets must be manually created in the namespace
    ##
    tls:
      - secretName: artifactory-tls
        hosts:
          - artifactory.domain.com
```

### Ingress annotations

This example specifically enables Artifactory to work as a Docker Registry using the Repository Path method. See [Artifactory as Docker Registry](https://www.jfrog.com/confluence/display/RTF/Getting+Started+with+Artifactory+as+a+Docker+Registry) documentation for more information about this setup.

```yaml
ingress:
  enabled: true
  defaultBackend:
    enabled: false
  hosts:
    - myhost.example.com
  annotations:
    ingress.kubernetes.io/force-ssl-redirect: "true"
    ingress.kubernetes.io/proxy-body-size: "0"
    ingress.kubernetes.io/proxy-read-timeout: "600"
    ingress.kubernetes.io/proxy-send-timeout: "600"
    kubernetes.io/ingress.class: nginx
    nginx.ingress.kubernetes.io/configuration-snippet: |
      rewrite ^/(v2)/token /artifactory/api/docker/null/v2/token;
      rewrite ^/(v2)/([^\/]*)/(.*) /artifactory/api/docker/$2/$1/$3;
    nginx.ingress.kubernetes.io/proxy-body-size: "0"
  tls:
    - hosts:
      - "myhost.example.com"
```

### Ingress additional rules

You have the option to add additional ingress rules to the Artifactory ingress. An example for this use case can be routing the /xray path to Xray.
In order to do that, simply add the following to a `artifactory-values.yaml` file:
```yaml
ingress:
  enabled: true

  defaultBackend:
    enabled: false

  annotations:
    kubernetes.io/ingress.class: nginx
    nginx.ingress.kubernetes.io/configuration-snippet: |
      rewrite "(?i)/xray(/|$)(.*)" /$2 break;

  additionalRules: |
    - host: <MY_HOSTNAME>
      http:
        paths:
          - path: /
            backend:
              serviceName: <XRAY_SERVER_SERVICE_NAME>
              servicePort: <XRAY_SERVER_SERVICE_PORT>
          - path: /xray
            backend:
              serviceName: <XRAY_SERVER_SERVICE_NAME>
              servicePort: <XRAY_SERVER_SERVICE_PORT>
          - path: /artifactory
            backend:
              serviceName: {{ template "artifactory.nginx.fullname" . }}
              servicePort: {{ .Values.nginx.externalPortHttp }}
``` 

and running:
```bash
helm upgrade --install xray jfrog/artifactory -f artifactory-values.yaml
```

## Useful links
https://www.jfrog.com
https://www.jfrog.com/confluence/
