# JFrog Artifactory High Availability Helm Chart

## Prerequisites Details

* Kubernetes 1.8+
* Artifactory HA license

## Chart Details
This chart will do the following:

* Deploy Artifactory highly available cluster. 1 primary node and 2 member nodes.
* Deploy a PostgreSQL database
* Deploy an Nginx server

## Artifactory HA architecture
The Artifactory HA cluster in this chart is made up of
- A single primary node
- Two member nodes, which can be resized at will

Load balancing is done to the member nodes only.
This leaves the primary node free to handle jobs and tasks and not be interrupted by inbound traffic.
> This can be controlled by the parameter `artifactory.service.pool`.

## Installing the Chart

### Add JFrog Helm repository
Before installing JFrog helm charts, you need to add the [JFrog helm repository](https://charts.jfrog.io/) to your helm client
```bash
helm repo add jfrog https://charts.jfrog.io
```

### Install Chart
To install the chart with the release name `artifactory-ha`:
```bash
helm install --name artifactory-ha jfrog/artifactory-ha
```

### Deploying Artifactory with replicator
The [Artifactory replicator](https://www.jfrog.com/confluence/display/RTF/Replicator) is used with an [Enterprise Plus](https://www.jfrog.com/confluence/display/EP/Welcome+to+JFrog+Enterprise+Plus) license.
```bash
## Artifactory replicator is disabled by default. When the replicator is enabled, the replicator.publicUrl parameter is required. To enable it use the following:
helm install --name artifactory --set artifactory.replicator.enabled=true --set artifactory.replicator.publicUrl=<artifactory_url>:<replicator_port> jfrog/artifactory-ha
```

### Accessing Artifactory
**NOTE:** It might take a few minutes for Artifactory's public IP to become available, and the nodes to complete initial setup.
Follow the instructions outputted by the install command to get the Artifactory IP and URL to access it.

### Updating Artifactory
Once you have a new chart version, you can update your deployment with
```bash
helm upgrade artifactory-ha jfrog/artifactory-ha
```

If artifactory was installed without providing a value to postgresql.postgresPassword (a password was autogenerated), follow these instructions:
1. Get the current password by running:
```bash
POSTGRES_PASSWORD=$(kubectl get secret -n <namespace> <myrelease>-postgresql -o jsonpath="{.data.postgres-password}" | base64 --decode)
```
2. Upgrade the release by passing the previously auto-generated secret:
```bash
helm upgrade <myrelease> jfrog/artifactory-ha --set postgresql.postgresPassword=${POSTGRES_PASSWORD}
```

This will apply any configuration changes on your existing deployment.

### Artifactory memory and CPU resources
The Artifactory HA Helm chart comes with support for configured resource requests and limits to all pods. By default, these settings are commented out.
It is **highly** recommended to set these so you have full control of the allocated resources and limits.

See more information on [setting resources for your Artifactory based on planned usage](https://www.jfrog.com/confluence/display/RTF/System+Requirements#SystemRequirements-RecommendedHardware).

```bash
# Example of setting resource requests and limits to all pods (including passing java memory settings to Artifactory)
helm install --name artifactory-ha \
               --set artifactory.primary.resources.requests.cpu="500m" \
               --set artifactory.primary.resources.limits.cpu="2" \
               --set artifactory.primary.resources.requests.memory="1Gi" \
               --set artifactory.primary.resources.limits.memory="4Gi" \
               --set artifactory.primary.javaOpts.xms="1g" \
               --set artifactory.primary.javaOpts.xmx="4g" \
               --set artifactory.node.resources.requests.cpu="500m" \
               --set artifactory.node.resources.limits.cpu="2" \
               --set artifactory.node.resources.requests.memory="1Gi" \
               --set artifactory.node.resources.limits.memory="4Gi" \
               --set artifactory.node.javaOpts.xms="1g" \
               --set artifactory.node.javaOpts.xmx="4g" \
               --set postgresql.resources.requests.cpu="200m" \
               --set postgresql.resources.limits.cpu="1" \
               --set postgresql.resources.requests.memory="500Mi" \
               --set postgresql.resources.limits.memory="1Gi" \
               --set nginx.resources.requests.cpu="100m" \
               --set nginx.resources.limits.cpu="250m" \
               --set nginx.resources.requests.memory="250Mi" \
               --set nginx.resources.limits.memory="500Mi" \
               jfrog/artifactory-ha
```
> Artifactory java memory parameters can (and should) also be set to match the allocated resources with `artifactory.[primary|node].javaOpts.xms` and `artifactory.[primary|node].javaOpts.xmx`.

Get more details on configuring Artifactory in the [official documentation](https://www.jfrog.com/confluence/).

### Artifactory storage
Artifactory HA support a wide range of storage back ends. You can see more details on [Artifactory HA storage options](https://www.jfrog.com/confluence/display/RTF/HA+Installation+and+Setup#HAInstallationandSetup-SettingUpYourStorageConfiguration)

In this chart, you set the type of storage you want with `artifactory.persistence.type` and pass the required configuration settings.
The default storage in this chart is the `file-system` replication, where the data is replicated to all nodes.

> **IMPORTANT:** All storage configurations (except NFS) come with a default `artifactory.persistence.redundancy` parameter.
This is used to set how many replicas of a binary should be stored in the cluster's nodes.
Once this value is set on initial deployment, you can not update it using helm.
It is recommended to set this to a number greater than half of your cluster's size, and never scale your cluster down to a size smaller than this number.

#### Existing volume claim

###### Primary node
In order to use an existing volume claim for the Artifactory primary storage, you need to:
- Create a persistent volume claim by the name `volume-<release-name>-artifactory-ha-primary-0` e.g `volume-myrelease-artifactory-ha-primary-0`
- Pass a parameter to `helm install` and `helm upgrade`
```bash
...
--set artifactory.primary.persistence.existingClaim=true
```

###### Member nodes
In order to use an existing volume claim for the Artifactory member nodes storage, you need to:
- Create persistent volume claims according to the number of replicas defined at `artifactory.node.replicaCount` by the names `volume-<release-name>-artifactory-ha-member-<ordinal-number>`, e.g `volume-myrelease-artifactory-ha-member-0` and `volume-myrelease-artifactory-ha-primary-1`.
- Pass a parameter to `helm install` and `helm upgrade`
```bash
...
--set artifactory.node.persistence.existingClaim=true
```

#### Existing shared volume claim

In order to use an existing claim (for data and backup) that is to be shared across all nodes, you need to:

- Create PVCs with ReadWriteMany that match the naming conventions:
```
  {{ template "artifactory-ha.fullname" . }}-data-pvc-<claim-ordinal>
  {{ template "artifactory-ha.fullname" . }}-backup-pvc-<claim-ordinal>
```
An example that shows 2 existing claims to be used:
```
  myexample-artifactory-ha-data-pvc-0
  myexample-artifactory-ha-backup-pvc-0
  myexample-artifactory-ha-data-pvc-1
  myexample-artifactory-ha-backup-pvc-1
```
- Set the artifactory.persistence.file-system.existingSharedClaim.enabled in values.yaml to true:
```
-- set artifactory.persistence.fileSystem.existingSharedClaim.enabled=true
```

#### NFS
To use an NFS server as your cluster's storage, you need to
- Setup an NFS server. Get its IP as `NFS_IP`
- Create a `data` and `backup` directories on the NFS exported directory with write permissions to all
- Pass NFS parameters to `helm install` and `helm upgrade`
```bash
...
--set artifactory.persistence.type=nfs \
--set artifactory.persistence.nfs.ip=${NFS_IP} \
...
```

#### Google Storage
To use a Google Storage bucket as the cluster's filestore. See [Google Storage Binary Provider](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore#ConfiguringtheFilestore-GoogleStorageBinaryProvider)
- Pass Google Storage parameters to `helm install` and `helm upgrade`
```bash
...
--set artifactory.persistence.type=google-storage \
--set artifactory.persistence.googleStorage.identity=${GCP_ID} \
--set artifactory.persistence.googleStorage.credential=${GCP_KEY} \
...
```

#### AWS S3
To use an AWS S3 bucket as the cluster's filestore. See [S3 Binary Provider](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore#ConfiguringtheFilestore-S3BinaryProvider)
- Pass AWS S3 parameters to `helm install` and `helm upgrade`
```bash
...
# With explicit credentials:
--set artifactory.persistence.type=aws-s3 \
--set artifactory.persistence.awsS3.endpoint=${AWS_S3_ENDPOINT} \
--set artifactory.persistence.awsS3.region=${AWS_REGION} \
--set artifactory.persistence.awsS3.identity=${AWS_ACCESS_KEY_ID} \
--set artifactory.persistence.awsS3.credential=${AWS_SECRET_ACCESS_KEY} \
...

...
# With using existing IAM role
--set artifactory.persistence.type=aws-s3 \
--set artifactory.persistence.awsS3.endpoint=${AWS_S3_ENDPOINT} \
--set artifactory.persistence.awsS3.region=${AWS_REGION} \
--set artifactory.persistence.awsS3.roleName=${AWS_ROLE_NAME} \
...
```
**NOTE:** Make sure S3 `endpoint` and `region` match. See [AWS documentation on endpoint](https://docs.aws.amazon.com/general/latest/gr/rande.html)

#### Microsoft Azure Blob Storage
To use Azure Blob Storage as the cluster's filestore. See [Azure Blob Storage Binary Provider](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Filestore#ConfiguringtheFilestore-AzureBlobStorageClusterBinaryProvider)
- Pass Azure Blob Storage parameters to `helm install` and `helm upgrade`
```bash
...
--set artifactory.persistence.type=azure-blob \
--set artifactory.persistence.azureBlob.accountName=${AZURE_ACCOUNT_NAME} \
--set artifactory.persistence.azureBlob.accountKey=${AZURE_ACCOUNT_KEY} \
--set artifactory.persistence.azureBlob.endpoint=${AZURE_ENDPOINT} \
--set artifactory.persistence.azureBlob.containerName=${AZURE_CONTAINER_NAME} \
...
```

### Create a unique Master Key
Artifactory HA cluster requires a unique master key. By default the chart has one set in values.yaml (`artifactory.masterKey`).

**This key is for demo purpose and should not be used in a production environment!**

You should generate a unique one and pass it to the template at install/upgrade time.
```bash
# Create a key
export MASTER_KEY=$(openssl rand -hex 32)
echo ${MASTER_KEY}

# Pass the created master key to helm
helm install --name artifactory-ha --set artifactory.masterKey=${MASTER_KEY} jfrog/artifactory-ha
```

Alternatively, you can create a secret containing the master key manually and pass it to the template at install/upgrade time.
```bash
# Create a key
export MASTER_KEY=$(openssl rand -hex 32)
echo ${MASTER_KEY}

# Create a secret containing the key. The key in the secret must be named master-key
kubectl create secret generic my-secret --from-literal=master-key=${MASTER_KEY}

# Pass the created secret to helm
helm install --name artifactory-ha --set artifactory.masterKeySecretName=my-secret jfrog/artifactory-ha
```
**NOTE:** In either case, make sure to pass the same master key on all future calls to `helm install` and `helm upgrade`! In the first case, this means always passing `--set artifactory.masterKey=${MASTER_KEY}`. In the second, this means always passing `--set artifactory.masterKeySecretName=my-secret` and ensuring the contents of the secret remain unchanged.

### Install Artifactory HA license
For activating Artifactory HA, you must install an appropriate license. There are two ways to manage the license. **Artifactory UI** or a **Kubernetes Secret**.

The easier and recommended way is the **Artifactory UI**. Using the **Kubernetes Secret** is for advanced users and is better suited for automation.

**IMPORTANT:** You should use only one of the following methods. Switching between them while a cluster is running might disable your Artifactory HA cluster!

##### Artifactory UI
Once primary cluster is running, open Artifactory UI and insert the license(s) in the UI. See [HA installation and setup](https://www.jfrog.com/confluence/display/RTF/HA+Installation+and+Setup) for more details

##### Kubernetes Secret
You can deploy the Artifactory license(s) as a [Kubernetes secret](https://kubernetes.io/docs/concepts/configuration/secret/).
Prepare a text file with the license(s) written in it. If writing multiple licenses (must be in the same file), it's important to put **two new lines between each license block**!
```bash
# Create the Kubernetes secret (assuming the local license file is 'art.lic')
kubectl create secret generic artifactory-cluster-license --from-file=./art.lic

# Pass the license to helm
helm install --name artifactory-ha --set artifactory.license.secret=artifactory-cluster-license,artifactory.license.dataKey=art.lic jfrog/artifactory-ha
```
**NOTE:** This method is relevant for initial deployment only! Once Artifactory is deployed, you should not keep passing these parameters as the license is already persisted into Artifactory's storage (they will be ignored).
Updating the license should be done via Artifactory UI or REST API.

##### Create the secret as part of the helm release
values.yaml
```yaml
artifactory:
  license:
    licenseKey: |-
      <LICENSE_KEY1>
      
      
      <LICENSE_KEY2>
      
      
      <LICENSE_KEY3>
```

```bash
helm install --name artifactory-ha -f values.yaml jfrog/artifactory-ha
```
**NOTE:** This method is relevant for initial deployment only! Once Artifactory is deployed, you should not keep passing these parameters as the license is already persisted into Artifactory's storage (they will be ignored).
Updating the license should be done via Artifactory UI or REST API.
If you want to keep managing the artifactory license using the same method, you can use the copyOnEveryStartup example shown in the values.yaml file

### Configure NetworkPolicy

NetworkPolicy specifies what ingress and egress is allowed in this namespace. It is encouraged to be more specific whenever possible to increase security of the system.

In the `networkpolicy` section of values.yaml you can specify a list of NetworkPolicy objects.

For podSelector, ingress and egress, if nothing is provided then a default `- {}` is applied which is to allow everything.

A full (but very wide open) example that results in 2 NetworkPolicy objects being created:
```
networkpolicy:
  # Allows all ingress and egress to/from artifactory primary and member pods.
  - name: artifactory
    podSelector:
      matchLabels:
        app: artifactory-ha
    egress:
    - {}
    ingress:
    - {}
  # Allows connectivity from artifactory-ha pods to postgresql pods, but no traffic leaving postgresql pod.
  - name: postgres
    podSelector:
      matchLabels:
        app: postgresql
    ingress:
    - from:
      - podSelector:
          matchLabels:
            app: artifactory-ha
```

### Bootstrapping Artifactory
**IMPORTANT:** Bootstrapping Artifactory needs license. Pass license as shown in above section.

* User guide to [bootstrap Artifactory Global Configuration](https://www.jfrog.com/confluence/display/RTF/Configuration+Files#ConfigurationFiles-BootstrappingtheGlobalConfiguration)
* User guide to [bootstrap Artifactory Security Configuration](https://www.jfrog.com/confluence/display/RTF/Configuration+Files#ConfigurationFiles-BootstrappingtheSecurityConfiguration)

Create `bootstrap-config.yaml` with artifactory.config.import.xml and security.import.xml as shown below:
```
apiVersion: v1
kind: ConfigMap
metadata:
  name: my-release-bootstrap-config
data:
  artifactory.config.import.xml: |
    <config contents>
  security.import.xml: |
    <config contents>
```

Create configMap in Kubernetes:
```bash
kubectl apply -f bootstrap-config.yaml
```

#### Pass the configMap to helm
```bash
helm install --name artifactory-ha --set artifactory.license.secret=artifactory-cluster-license,artifactory.license.dataKey=art.lic,artifactory.configMapName=my-release-bootstrap-config jfrog/artifactory-ha
```

### Use custom nginx.conf with Nginx

Steps to create configMap with nginx.conf
* Create `nginx.conf` file.
```bash
kubectl create configmap nginx-config --from-file=nginx.conf
```
* Pass configMap to helm install
```bash
helm install --name artifactory-ha --set nginx.customConfigMap=nginx-config jfrog/artifactory-ha
```

### Scaling your Artifactory cluster
A key feature in Artifactory HA is the ability to set an initial cluster size with `--set artifactory.node.replicaCount=${CLUSTER_SIZE}` and if needed, resize it.

##### Before scaling
**IMPORTANT:** When scaling, you need to explicitly pass the database password if it's an auto generated one (this is the default with the enclosed PostgreSQL helm chart).

Get the current database password
```bash
export DB_PASSWORD=$(kubectl get $(kubectl get secret -o name | grep postgresql) -o jsonpath="{.data.postgres-password}" | base64 --decode)
```
Use `--set postgresql.postgresPassword=${DB_PASSWORD}` with every scale action to prevent a miss configured cluster!

##### Scale up
Let's assume you have a cluster with **2** member nodes, and you want to scale up to **3** member nodes (a total of 4 nodes).
```bash
# Scale to 4 nodes (1 primary and 3 member nodes)
helm upgrade --install artifactory-ha --set artifactory.node.replicaCount=3 --set postgresql.postgresPassword=${DB_PASSWORD} jfrog/artifactory-ha
```

##### Scale down
Let's assume you have a cluster with **3** member nodes, and you want to scale down to **2** member node.

```bash
# Scale down to 2 member nodes
helm upgrade --install artifactory-ha --set artifactory.node.replicaCount=2 --set postgresql.postgresPassword=${DB_PASSWORD} jfrog/artifactory-ha
```
- **NOTE:** Since Artifactory is running as a Kubernetes Stateful Set, the removal of the node will **not** remove the persistent volume. You need to explicitly remove it
```bash
# List PVCs
kubectl get pvc

# Remove the PVC with highest ordinal!
# In this example, the highest node ordinal was 2, so need to remove its storage.
kubectl delete pvc volume-artifactory-node-2
```

### Use an external Database
There are cases where you will want to use a different database and not the enclosed **PostgreSQL**.
See more details on [configuring the database](https://www.jfrog.com/confluence/display/RTF/Configuring+the+Database)
> The official Artifactory Docker images include the PostgreSQL database driver.
> For other database types, you will have to add the relevant database driver to Artifactory's tomcat/lib

This can be done with the following parameters
```bash
# Make sure your Artifactory Docker image has the MySQL database driver in it
...
--set postgresql.enabled=false \
--set artifactory.preStartCommand="wget -O /opt/jfrog/artifactory/tomcat/lib/mysql-connector-java-5.1.41.jar https://jcenter.bintray.com/mysql/mysql-connector-java/5.1.41/mysql-connector-java-5.1.41.jar" \
--set database.type=mysql \
--set database.host=${DB_HOST} \
--set database.port=${DB_PORT} \
--set database.user=${DB_USER} \
--set database.password=${DB_PASSWORD} \
...
```
**NOTE:** You must set `postgresql.enabled=false` in order for the chart to use the `database.*` parameters. Without it, they will be ignored!

If you store your database credentials in a pre-existing Kubernetes `Secret`, you can specify them via `database.secrets` instead of `database.user` and `database.password`:
```bash
# Create a secret containing the database credentials
kubectl create secret generic my-secret --from-literal=user=${DB_USER} --from-literal=password=${DB_PASSWORD}
...
--set postgresql.enabled=false \
--set database.secrets.user.name=my-secret \
--set database.secrets.user.key=user \
--set database.secrets.password.name=my-secret \
--set database.secrets.password.key=password \
...
```

### Deleting Artifactory
To delete the Artifactory HA cluster
```bash
helm delete --purge artifactory-ha
```
This will completely delete your Artifactory HA cluster.
**NOTE:** Since Artifactory is running as Kubernetes Stateful Sets, the removal of the helm release will **not** remove the persistent volumes. You need to explicitly remove them
```bash
kubectl delete pvc -l release=artifactory-ha
```
See more details in the official [Kubernetes Stateful Set removal page](https://kubernetes.io/docs/tasks/run-application/delete-stateful-set/)

### Custom Docker registry for your images
If you need to pull your Docker images from a private registry (for example, when you have a custom image with a MySQL database driver), you need to create a
[Kubernetes Docker registry secret](https://kubernetes.io/docs/tasks/configure-pod-container/pull-image-private-registry/) and pass it to helm
```bash
# Create a Docker registry secret called 'regsecret'
kubectl create secret docker-registry regsecret --docker-server=${DOCKER_REGISTRY} --docker-username=${DOCKER_USER} --docker-password=${DOCKER_PASS} --docker-email=${DOCKER_EMAIL}
```
Once created, you pass it to `helm`
```bash
helm install --name artifactory-ha --set imagePullSecrets=regsecret jfrog/artifactory-ha
```

### Logger sidecars
This chart provides the option to add sidecars to tail various logs from Artifactory. See the available values in [values.yaml](values.yaml)

Get list of containers in the pod
```bash
kubectl get pods -n <NAMESPACE> <POD_NAME> -o jsonpath='{.spec.containers[*].name}' | tr ' ' '\n'
```

View specific log
```bash
kubectl logs -n <NAMESPACE> <POD_NAME> -c <LOG_CONTAINER_NAME>
```


### Custom init containers
There are cases where a special, unsupported init processes is needed like checking something on the file system or testing something before spinning up the main container.

For this, there is a section for writing a custom init container in the [values.yaml](values.yaml). By default it's commented out
```
artifactory:
  ## Add custom init containers
  customInitContainers: |
    ## Init containers template goes here ##
```

### Custom sidecar containers
There are cases where an extra sidecar container is needed. For example monitoring agents or log collection.

For this, there is a section for writing a custom sidecar container in the [values.yaml](values.yaml). By default it's commented out
```
artifactory:
  ## Add custom sidecar containers
  customSidecarContainers: |
    ## Sidecar containers template goes here ##
```

You can configure the sidecar to run as a custom user if needed by setting the following in the container template
```
  # Example of running container as root (id 0)
  securityContext:
    runAsUser: 0
    fsGroup: 0
```

### Custom volumes
If you need to use a custom volume in a custom init or sidecar container, you can use this option.

For this, there is a section for defining custom volumes in the [values.yaml](values.yaml). By default it's commented out
```
artifactory:
  ## Add custom volumes
  customVolumes: |
    ## Custom volume comes here ##
```

### Add Artifactory User Plugin during installation
If you need to add [Artifactory User Plugin](https://github.com/jfrog/artifactory-user-plugins), you can use this option.

Create a secret with [Artifactory User Plugin](https://github.com/jfrog/artifactory-user-plugins) by following command:
```
# Secret with single user plugin
kubectl  create secret generic archive-old-artifacts --from-file=archiveOldArtifacts.groovy --namespace=artifactory-ha 

# Secret with single user plugin with configuration file
kubectl  create secret generic webhook --from-file=webhook.groovy  --from-file=webhook.config.json.sample --namespace=artifactory-ha
```

Add plugin secret names to `plugins.yaml` as following:
```yaml
artifactory:
  userPluginSecrets:
    - archive-old-artifacts
    - webhook
```

You can now pass the created `plugins.yaml` file to helm install command to deploy Artifactory with user plugins as follows:
```
helm install --name artifactory-ha -f plugins.yaml jfrog/artifactory-ha
```

Alternatively, you may be in a situation in which you would like to create a secret in a Helm chart that depends on this chart. In this scenario, the name of the secret is likely dynamically generated via template functions, so passing a statically named secret isn't possible. In this case, the chart supports evaluating strings as templates via the [`tpl`](https://helm.sh/docs/charts_tips_and_tricks/#using-the-tpl-function) function - simply pass the raw string containing the templating language used to name your secret as a value instead by adding the following to your chart's `values.yaml` file:
```yaml
artifactory-ha: # Name of the artifactory-ha dependency
  artifactory:
    userPluginSecrets:
      - '{{ template "my-chart.fullname" . }}'
```
NOTE: By defining userPluginSecrets, this overrides any pre-defined plugins from the container image that are stored in /tmp/plugins.  At this time [artifactory-pro:6.9.0](https://bintray.com/jfrog/artifactory-pro) is distributed with `internalUser.groovy` plugin.  If you need this plugin in addition to your user plugins, you should include these additional plugins as part of your userPluginSecrets.

## Configuration
The following table lists the configurable parameters of the artifactory chart and their default values.

|         Parameter            |           Description             |                         Default                       |
|------------------------------|-----------------------------------|-------------------------------------------------------|
| `imagePullSecrets`           | Docker registry pull secret       |                                                       |
| `serviceAccount.create`   | Specifies whether a ServiceAccount should be created | `true`                                |
| `serviceAccount.name`     | The name of the ServiceAccount to create             | Generated using the fullname template |
| `rbac.create`             | Specifies whether RBAC resources should be created   | `true`                                |
| `rbac.role.rules`         | Rules to create                   | `[]`                                                     |
| `logger.image.repository` | repository for logger image       | `busybox`                                                |
| `logger.image.tag`        | tag for logger image              | `1.30`                                                   |
| `artifactory.name`                   | Artifactory name                     | `artifactory`                              |
| `artifactory.image.pullPolicy`       | Container pull policy                | `IfNotPresent`                             |
| `artifactory.image.repository`       | Container image                      | `docker.bintray.io/jfrog/artifactory-pro`  |
| `artifactory.image.version`          | Container image tag                  | `.Chart.AppVersion`                        |
| `artifactory.loggers`             | Artifactory loggers (see values.yaml for possible values) | `[]`                     |
| `artifactory.catalinaLoggers`     | Artifactory Tomcat loggers (see values.yaml for possible values) | `[]`              |
| `artifactory.customInitContainers`| Custom init containers                  |                                            |
| `artifactory.customSidecarContainers`| Custom sidecar containers            |                                            |
| `artifactory.customVolumes`       | Custom volumes                    |                                                  |
| `artifactory.customVolumeMounts`  | Custom Artifactory volumeMounts   |                                                  |
| `artifactory.userPluginSecrets`   | Array of secret names for Artifactory user plugins |                                 |
| `artifactory.masterKey`           | Artifactory Master Key. Can be generated with `openssl rand -hex 32` |`FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF`|
| `artifactory.masterKeySecretName` | Artifactory Master Key secret name                     |                             |
| `artifactory.accessAdmin.password`               | Artifactory access-admin password to be set upon startup|             |
| `artifactory.accessAdmin.secret`                 | Artifactory access-admin secret name |                                |
| `artifactory.accessAdmin.dataKey`                | Artifactory access-admin secret data key |                            |
| `artifactory.preStartCommand`                    | Command to run before entrypoint starts |                             |
| `artifactory.postStartCommand`                   | Command to run after container starts   |                             |
| `artifactory.license.licenseKey` | Artifactory license key. Providing the license key as a parameter will cause a secret containing the license key to be created as part of the release. Use either this setting or the license.secret and license.dataKey. If you use both, the latter will be used.  |           |
| `artifactory.license.secret` | Artifactory license secret name              |                                            |
| `artifactory.license.dataKey`| Artifactory license secret data key          |                                            |
| `artifactory.service.name`   | Artifactory service name to be set in Nginx configuration | `artifactory`                 |
| `artifactory.service.type`   | Artifactory service type                                  | `ClusterIP`                   |
| `artifactory.service.pool`   | Artifactory instances to be in the load balancing pool. `members` or `all` | `members`    |
| `artifactory.externalPort`   | Artifactory service external port                         | `8081`                        |
| `artifactory.internalPort`   | Artifactory service internal port                         | `8081`                        |
| `artifactory.internalPortReplicator` | Replicator service internal port | `6061`   |
| `artifactory.externalPortReplicator` | Replicator service external port | `6061`   |
| `artifactory.extraEnvironmentVariables`          | Extra environment variables to pass to Artifactory. Supports evaluating strings as templates via the [`tpl`](https://helm.sh/docs/charts_tips_and_tricks/#using-the-tpl-function) function. See [documentation](https://www.jfrog.com/confluence/display/RTF/Installing+with+Docker#InstallingwithDocker-SupportedEnvironmentVariables) |   |
| `artifactory.livenessProbe.enabled`              | Enable liveness probe                     |  `true`                                               |
| `artifactory.livenessProbe.path`                 | liveness probe HTTP Get path              |  `/artifactory/webapp/#/login`                        |
| `artifactory.livenessProbe.initialDelaySeconds`  | Delay before liveness probe is initiated  | 180                                                   |
| `artifactory.livenessProbe.periodSeconds`        | How often to perform the probe            | 10                                                    |
| `artifactory.livenessProbe.timeoutSeconds`       | When the probe times out                  | 10                                                    |
| `artifactory.livenessProbe.successThreshold`     | Minimum consecutive successes for the probe to be considered successful after having failed. | 1  |
| `artifactory.livenessProbe.failureThreshold`     | Minimum consecutive failures for the probe to be considered failed after having succeeded.   | 10 |
| `artifactory.readinessProbe.enabled`              | would you like a readinessProbe to be enabled           |  `true`                                |
| `artifactory.readinessProbe.path`                      | readiness probe HTTP Get path                           |  `/artifactory/webapp/#/login`           |
| `artifactory.readinessProbe.initialDelaySeconds` | Delay before readiness probe is initiated | 60                                                    |
| `artifactory.readinessProbe.periodSeconds`       | How often to perform the probe            | 10                                                    |
| `artifactory.readinessProbe.timeoutSeconds`      | When the probe times out                  | 10                                                    |
| `artifactory.readinessProbe.successThreshold`    | Minimum consecutive successes for the probe to be considered successful after having failed. | 1  |
| `artifactory.readinessProbe.failureThreshold`    | Minimum consecutive failures for the probe to be considered failed after having succeeded.   | 10 |
| `artifactory.copyOnEveryStartup`     | List of files to copy on startup from source (which is absolute) to target (which is relative to ARTIFACTORY_HOME   |  |
| `artifactory.persistence.mountPath`    | Artifactory persistence volume mount path           | `"/var/opt/jfrog/artifactory"`  |
| `artifactory.persistence.enabled`      | Artifactory persistence volume enabled              | `true`                          |
| `artifactory.persistence.accessMode`   | Artifactory persistence volume access mode          | `ReadWriteOnce`                 |
| `artifactory.persistence.size`         | Artifactory persistence or local volume size        | `200Gi`                         |
| `artifactory.persistence.maxCacheSize` | Artifactory cache-fs provider maxCacheSize in bytes | `50000000000`                   |
| `artifactory.persistence.type`         | Artifactory HA storage type                         | `file-system`                   |
| `artifactory.persistence.redundancy`   | Artifactory HA storage redundancy                   | `3`                             |
| `artifactory.persistence.nfs.ip`            | NFS server IP                        |                                     |
| `artifactory.persistence.nfs.haDataMount`   | NFS data directory                   | `/data`                             |
| `artifactory.persistence.nfs.haBackupMount` | NFS backup directory                 | `/backup`                           |
| `artifactory.persistence.nfs.dataDir`       | HA data directory                    | `/var/opt/jfrog/artifactory-ha`     |
| `artifactory.persistence.nfs.backupDir`     | HA backup directory                  | `/var/opt/jfrog/artifactory-backup` |
| `artifactory.persistence.nfs.capacity`      | NFS PVC size                         | `200Gi`                             |
| `artifactory.persistence.eventual.numberOfThreads`  | Eventual number of threads   | `10`                                |
| `artifactory.persistence.googleStorage.bucketName`  | Google Storage bucket name          | `artifactory-ha`             |
| `artifactory.persistence.googleStorage.identity`    | Google Storage service account id   |                              |
| `artifactory.persistence.googleStorage.credential`  | Google Storage service account key  |                              |
| `artifactory.persistence.googleStorage.path`        | Google Storage path in bucket       | `artifactory-ha/filestore`   |
| `artifactory.persistence.googleStorage.bucketExists`| Google Storage bucket exists therefore does not need to be created.| `false`             |
| `artifactory.persistence.awsS3.bucketName`          | AWS S3 bucket name                     | `artifactory-ha`             |
| `artifactory.persistence.awsS3.endpoint`            | AWS S3 bucket endpoint                 | See https://docs.aws.amazon.com/general/latest/gr/rande.html |
| `artifactory.persistence.awsS3.region`              | AWS S3 bucket region                   |                              |
| `artifactory.persistence.awsS3.roleName`            | AWS S3 IAM role name                   |                             |
| `artifactory.persistence.awsS3.identity`            | AWS S3 AWS_ACCESS_KEY_ID               |                              |
| `artifactory.persistence.awsS3.credential`          | AWS S3 AWS_SECRET_ACCESS_KEY           |                              |
| `artifactory.persistence.awsS3.properties`          | AWS S3 additional properties           |                              |
| `artifactory.persistence.awsS3.path`                | AWS S3 path in bucket                  | `artifactory-ha/filestore`   |
| `artifactory.persistence.awsS3.refreshCredentials`  | AWS S3 renew credentials on expiration | `true` (When roleName is used, this parameter will be set to true) |
| `artifactory.persistence.awsS3.testConnection`      | AWS S3 test connection on start up     | `false`                      |
| `artifactory.persistence.awsS3.s3AwsVersion`        | AWS S3 signature version               | `AWS4-HMAC-SHA256`           |
| `artifactory.persistence.azureBlob.accountName`     | Azure Blob Storage account name        | ``                        |
| `artifactory.persistence.azureBlob.accountKey`      | Azure Blob Storage account key         | ``                        |
| `artifactory.persistence.azureBlob.endpoint`        | Azure Blob Storage endpoint            | ``                        |
| `artifactory.persistence.azureBlob.containerName`   | Azure Blob Storage container name      | ``                        |
| `artifactory.persistence.azureBlob.testConnection`  | Azure Blob Storage test connection     | `false`                   |
| `artifactory.persistence.fileStorage.existingSharedClaim` | Enable using an existing shared pvc | `false`                             |
| `artifactory.persistence.fileStorage.dataDir`             | HA data directory                   | `/var/opt/jfrog/artifactory/artifactory-data`     |
| `artifactory.persistence.fileStorage.backupDir`           | HA backup directory                 | `/var/opt/jfrog/artifactory-backup` |
| `artifactory.javaOpts.other`                        | Artifactory additional java options (for all nodes) |              |
| `artifactory.replicator.enabled`                    | Enable Artifactory Replicator          | `false`                   |
| `artifactory.replicator.publicUrl`              | Artifactory Replicator Public URL |                                    |
| `artifactory.primary.resources.requests.memory` | Artifactory primary node initial memory request  |                     |
| `artifactory.primary.resources.requests.cpu`    | Artifactory primary node initial cpu request     |                     |
| `artifactory.primary.resources.limits.memory`   | Artifactory primary node memory limit            |                     |
| `artifactory.primary.resources.limits.cpu`      | Artifactory primary node cpu limit               |                     |
| `artifactory.primary.javaOpts.xms`              | Artifactory primary node java Xms size           |                     |
| `artifactory.primary.javaOpts.xmx`              | Artifactory primary node java Xms size           |                     |
| `artifactory.primary.javaOpts.other`            | Artifactory primary node additional java options |                     |
| `artifactory.primary.persistence.existingClaim` | Whether to use an existing pvc for the primary node | `false`            |
| `artifactory.node.replicaCount`                 | Artifactory member node replica count            | `2`                 |
| `artifactory.node.minAvailable`                 | Artifactory member node min available count      | `1`                 |
| `artifactory.node.resources.requests.memory`    | Artifactory member node initial memory request   |                     |
| `artifactory.node.resources.requests.cpu`       | Artifactory member node initial cpu request      |                     |
| `artifactory.node.resources.limits.memory`      | Artifactory member node memory limit             |                     |
| `artifactory.node.resources.limits.cpu`         | Artifactory member node cpu limit                |                     |
| `artifactory.node.javaOpts.xms`                 | Artifactory member node java Xms size            |                     |
| `artifactory.node.javaOpts.xmx`                 | Artifactory member node java Xms size            |                     |
| `artifactory.node.javaOpts.other`               | Artifactory member node additional java options  |                     |
| `artifactory.node.persistence.existingClaim`    | Whether to use existing PVCs for the member nodes | `false`            |
| `artifactory.terminationGracePeriodSeconds`     | Termination grace period (seconds)               | `30s`               |
| `ingress.enabled`           | If true, Artifactory Ingress will be created | `false`                                     |
| `ingress.annotations`       | Artifactory Ingress annotations     | `{}`                                                 |
| `ingress.labels`       | Artifactory Ingress labels     | `{}`                                                           |
| `ingress.hosts`             | Artifactory Ingress hostnames       | `[]`                                                 |
| `ingress.path`              | Artifactory Ingress path            | `/`                                                  |
| `ingress.tls`               | Artifactory Ingress TLS configuration (YAML) | `[]`                                        |
| `ingress.defaultBackend.enabled` | If true, the default `backend` will be added using serviceName and servicePort | `true` |
| `ingress.annotations`       | Ingress annotations, which are written out if annotations section exists in values. Everything inside of the annotations section will appear verbatim inside the resulting manifest. See `Ingress annotations` section below for examples of how to leverage the annotations, specifically for how to enable docker authentication. |  |
| `nginx.enabled`             | Deploy nginx server                      | `true`                                          |
| `nginx.name`                | Nginx name                        | `nginx`                                                |
| `nginx.replicaCount`        | Nginx replica count               | `1`                                                    |
| `nginx.uid`                 | Nginx User Id                     | `104`                                                  |
| `nginx.gid`                 | Nginx Group Id                    | `107`                                                  |
| `nginx.image.repository`    | Container image                   | `docker.bintray.io/jfrog/nginx-artifactory-pro`        |
| `nginx.image.version`       | Container version                 | `.Chart.AppVersion`                                    |
| `nginx.image.pullPolicy`    | Container pull policy             | `IfNotPresent`                                         |
| `nginx.loggers`        | Artifactory loggers (see values.yaml for possible values) | `[]`                           |
| `nginx.service.type`        | Nginx service type                | `LoadBalancer`                                         |
| `nginx.service.loadBalancerSourceRanges`| Nginx service array of IP CIDR ranges to whitelist (only when service type is LoadBalancer) |        |
| `nginx.service.annotations` | Nginx service annotations           | `{}`                            |
| `nginx.service.externalTrafficPolicy`| Nginx service desires to route external traffic to node-local or cluster-wide endpoints. | `Cluster` |
| `nginx.loadBalancerIP`| Provide Static IP to configure with Nginx |                                 |
| `nginx.externalPortHttp` | Nginx service external port            | `80`                            |
| `nginx.internalPortHttp` | Nginx service internal port            | `80`                            |
| `nginx.externalPortHttps` | Nginx service external port           | `443`                           |
| `nginx.internalPortHttps` | Nginx service internal port           | `443`                           |
| `nginx.internalPortReplicator` | Replicator service internal port | `6061`                          |
| `nginx.externalPortReplicator` | Replicator service external port | `6061`                          |
| `nginx.livenessProbe.enabled`              | would you like a liveness Probe to be enabled          |  `true`                                  |
| `nginx.livenessProbe.path`                 | liveness probe HTTP Get path              |  `/artifactory/webapp/#/login`                    |
| `nginx.livenessProbe.initialDelaySeconds`  | Delay before liveness probe is initiated  | 100                                                   |
| `nginx.livenessProbe.periodSeconds`        | How often to perform the probe            | 10                                                    |
| `nginx.livenessProbe.timeoutSeconds`       | When the probe times out                  | 10                                                    |
| `nginx.livenessProbe.successThreshold`     | Minimum consecutive successes for the probe to be considered successful after having failed. | 1  |
| `nginx.livenessProbe.failureThreshold`     | Minimum consecutive failures for the probe to be considered failed after having succeeded.   | 10 |
| `nginx.readinessProbe.enabled`             | would you like a readinessProbe to be enabled           |  `true`                                 |
| `nginx.readinessProbe.path`                     | Readiness probe HTTP Get path                           |  `/artifactory/webapp/#/login` |
| `nginx.readinessProbe.initialDelaySeconds` | Delay before readiness probe is initiated | 60                                                    |
| `nginx.readinessProbe.periodSeconds`       | How often to perform the probe            | 10                                                    |
| `nginx.readinessProbe.timeoutSeconds`      | When the probe times out                  | 10                                                    |
| `nginx.readinessProbe.successThreshold`    | Minimum consecutive successes for the probe to be considered successful after having failed. | 1  |
| `nginx.readinessProbe.failureThreshold`    | Minimum consecutive failures for the probe to be considered failed after having succeeded.   | 10 |
| `nginx.tlsSecretName` |  SSL secret that will be used by the Nginx pod |                                                 |
| `nginx.env.ssl`                   | Nginx Environment enable ssl               | `true`                                  |
| `nginx.env.skipAutoConfigUpdate`  | Nginx Environment to disable auto configuration update | `false`                     |
| `nginx.customConfigMap`           | Nginx CustomeConfigMap name for `nginx.conf` | ` `                                   |
| `nginx.customArtifactoryConfigMap`| Nginx CustomeConfigMap name for `artifactory-ha.conf` | ` `                          |
| `nginx.resources.requests.memory` | Nginx initial memory request               | `250Mi`                                 |
| `nginx.resources.requests.cpu`    | Nginx initial cpu request                  | `100m`                                  |
| `nginx.resources.limits.memory`   | Nginx memory limit                         | `250Mi`                                 |
| `nginx.resources.limits.cpu`      | Nginx cpu limit                            | `500m`                                  |
| `nginx.persistence.mountPath` | Nginx persistence volume mount path | `"/var/opt/jfrog/nginx"`                           |
| `nginx.persistence.enabled` | Nginx persistence volume enabled. This is only available when the nginx.replicaCount is set to 1 | `false`                                                  |
| `nginx.persistence.accessMode` | Nginx persistence volume access mode | `ReadWriteOnce`                                  |
| `nginx.persistence.size` | Nginx persistence volume size | `5Gi`                                                         |
| `postgresql.enabled`              | Use enclosed PostgreSQL as database        | `true`                                  |
| `postgresql.imageTag`             | PostgreSQL version                         | `9.6.11`                                |
| `postgresql.postgresDatabase`     | PostgreSQL database name                   | `artifactory`                           |
| `postgresql.postgresUser`         | PostgreSQL database user                   | `artifactory`                           |
| `postgresql.postgresPassword`     | PostgreSQL database password               |                                         |
| `postgresql.persistence.enabled`  | PostgreSQL use persistent storage          | `true`                                  |
| `postgresql.persistence.size`     | PostgreSQL persistent storage size         | `50Gi`                                  |
| `postgresql.service.port`         | PostgreSQL database port                   | `5432`                                  |
| `postgresql.resources.requests.memory`    | PostgreSQL initial memory request  |                                         |
| `postgresql.resources.requests.cpu`       | PostgreSQL initial cpu request     |                                         |
| `postgresql.resources.limits.memory`      | PostgreSQL memory limit            |                                         |
| `postgresql.resources.limits.cpu`         | PostgreSQL cpu limit               |                                         |
| `database.type`                  | External database type (`postgresql`, `mysql`, `oracle` or `mssql`)  |                       |
| `database.host`                  | External database hostname                         |                                         |
| `database.port`                  | External database port                             |                                         |
| `database.url`                   | External database connection URL                   |                                         |
| `database.user`                  | External database username                         |                                         |
| `database.password`              | External database password                         |                                         |
| `database.secrets.user.name`     | External database username `Secret` name           |                                         |
| `database.secrets.user.key`      | External database username `Secret` key            |                                         |
| `database.secrets.password.name` | External database password `Secret` name           |                                         |
| `database.secrets.password.key`  | External database password `Secret` key            |                                         |
| `database.secrets.url.name     ` | External database url `Secret` name                |                                         |
| `database.secrets.url.key`       | External database url `Secret` key                 |                                         |
| `networkpolicy.name`             | Becomes part of the NetworkPolicy object name                                  | `artifactory`                           |
| `networkpolicy.podselector`      | Contains the YAML that specifies how to match pods. Usually using matchLabels. |                                         |
| `networkpolicy.ingress`          | YAML snippet containing to & from rules applied to incoming traffic            | `- {}` (open to all inbound traffic)    |
| `networkpolicy.egress`           | YAML snippet containing to & from rules applied to outgoing traffic            | `- {}` (open to all outbound traffic)   |


Specify each parameter using the `--set key=value[,key=value]` argument to `helm install`.

### Ingress and TLS
To get Helm to create an ingress object with a hostname, add these two lines to your Helm command:
```
helm install --name artifactory-ha \
  --set ingress.enabled=true \
  --set ingress.hosts[0]="artifactory.company.com" \
  --set artifactory.service.type=NodePort \
  --set nginx.enabled=false \
  jfrog/artifactory-ha
```

If your cluster allows automatic creation/retrieval of TLS certificates (e.g. [cert-manager](https://github.com/jetstack/cert-manager)), please refer to the documentation for that mechanism.

To manually configure TLS, first create/retrieve a key & certificate pair for the address(es) you wish to protect. Then create a TLS secret in the namespace:

```console
kubectl create secret tls artifactory-tls --cert=path/to/tls.cert --key=path/to/tls.key
```

Include the secret's name, along with the desired hostnames, in the Artifactory Ingress TLS section of your custom `values.yaml` file:

```
  ingress:
    ## If true, Artifactory Ingress will be created
    ##
    enabled: true

    ## Artifactory Ingress hostnames
    ## Must be provided if Ingress is enabled
    ##
    hosts:
      - artifactory.domain.com
    annotations:
      kubernetes.io/tls-acme: "true"
    ## Artifactory Ingress TLS configuration
    ## Secrets must be manually created in the namespace
    ##
    tls:
      - secretName: artifactory-tls
        hosts:
          - artifactory.domain.com
```

### Ingress annotations

This example specifically enables Artifactory to work as a Docker Registry using the Repository Path method. See [Artifactory as Docker Registry](https://www.jfrog.com/confluence/display/RTF/Getting+Started+with+Artifactory+as+a+Docker+Registry) documentation for more information about this setup.

```
ingress:
  enabled: true
  defaultBackend:
    enabled: false
  hosts:
    - myhost.example.com
  annotations:
    ingress.kubernetes.io/force-ssl-redirect: "true"
    ingress.kubernetes.io/proxy-body-size: "0"
    ingress.kubernetes.io/proxy-read-timeout: "600"
    ingress.kubernetes.io/proxy-send-timeout: "600"
    kubernetes.io/ingress.class: nginx
    nginx.ingress.kubernetes.io/configuration-snippet: |
      rewrite ^/(v2)/token /artifactory/api/docker/null/v2/token;
      rewrite ^/(v2)/([^\/]*)/(.*) /artifactory/api/docker/$2/$1/$3;
    nginx.ingress.kubernetes.io/proxy-body-size: "0"
  tls:
    - hosts:
      - "myhost.example.com"
```

## Useful links
- https://www.jfrog.com/confluence/display/EP/Getting+Started
- https://www.jfrog.com/confluence/display/RTF/Installing+Artifactory
- https://www.jfrog.com/confluence/
